<?php

namespace T4S\Rewriter\Util;

class UrlDecoder {
    /**
     * Decode Google News URLs and other encoded URLs
     */
    public static function decode(string $url): string {
        // Handle Google News URLs
        if (strpos($url, 'news.google.com') !== false) {
            // Extract the actual URL from Google News format
            preg_match('/articles\/CBMi(.+?)\?/', $url, $matches);
            if (!empty($matches[1])) {
                $encoded_url = $matches[1];
                // Base64 decode and handle URL safe characters
                $decoded_url = str_replace(['-', '_'], ['+', '/'], $encoded_url);
                $decoded_url = base64_decode($decoded_url);
                
                if ($decoded_url !== false) {
                    return $decoded_url;
                }
            }

            // Try to extract URL from parameters
            $params = parse_url($url, PHP_URL_QUERY);
            if ($params) {
                parse_str($params, $query);
                if (!empty($query['url'])) {
                    return $query['url'];
                }
            }
        }

        // Handle URLs with multiple redirects
        $response = wp_remote_head($url, [
            'redirection' => 5,
            'timeout' => 10
        ]);

        if (!is_wp_error($response) && !empty($response['http_response']->get_response_object()->url)) {
            return $response['http_response']->get_response_object()->url;
        }

        // Return original URL if no decoding was possible
        return $url;
    }

    /**
     * Check if URL needs decoding
     */
    public static function needsDecoding(string $url): bool {
        return strpos($url, 'news.google.com') !== false ||
               strpos($url, 'url=') !== false ||
               strpos($url, 'redirect=') !== false;
    }

    /**
     * Get final URL after following all redirects
     */
    public static function getFinalUrl(string $url): string {
        if (!self::needsDecoding($url)) {
            return $url;
        }

        try {
            $decoded_url = self::decode($url);
            
            // Verify decoded URL is valid
            if (filter_var($decoded_url, FILTER_VALIDATE_URL)) {
                return $decoded_url;
            }
        } catch (\Exception $e) {
            error_log('T4S Rewriter: URL decoding error: ' . $e->getMessage());
        }

        return $url;
    }
}