<?php

namespace T4S\Rewriter\Util;

use DOMDocument;
use DOMElement;

class BlockConverter
{
    /**
     * Convert HTML content to Gutenberg blocks
     */
    public function fromHtml(string $html): string
    {
        // Use UTF-8 for DOMDocument
        $dom = new DOMDocument('1.0', 'UTF-8');
        $dom->encoding = 'UTF-8';
        
        // Preserve UTF-8 content
        $html = '<!DOCTYPE html><html><head><meta charset="UTF-8"></head><body>' . $html . '</body></html>';
        
        // Disable libxml errors and use internal errors
        $previous = libxml_use_internal_errors(true);
        
        // Load HTML content
        $dom->loadHTML($html, LIBXML_HTML_NODEFDTD);
        
        // Clear errors
        libxml_clear_errors();
        libxml_use_internal_errors($previous);

        $blocks = [];
        
        // Get all direct children of the body
        $xpath = new \DOMXPath($dom);
        $bodyChildren = $xpath->query('/html/body/*');

        if (!$bodyChildren || $bodyChildren->length === 0) {
            return $html;
        }

        foreach ($bodyChildren as $node) {
            if ($node instanceof DOMElement) {
                $block = $this->convertElementToBlock($node);
                if ($block) {
                    $blocks[] = $block;
                }
            }
        }

        return implode("\n\n", array_filter($blocks));
    }

    /**
     * Convert a DOM element to a Gutenberg block
     */
    private function convertElementToBlock(DOMElement $element): ?string
    {
        switch ($element->tagName) {
            case 'p':
                // Check if paragraph contains an image
                $imgNode = null;
                $hasImage = false;
                $hasText = false;
                $textContent = '';

                foreach ($element->childNodes as $child) {
                    if ($child instanceof DOMElement && $child->tagName === 'img') {
                        $hasImage = true;
                        $imgNode = $child;
                    } else {
                        // Check if there's actual text content
                        $content = trim($child->textContent);
                        if (!empty($content)) {
                            $hasText = true;
                            $textContent = $content;
                        }
                    }
                }

                if ($hasImage) {
                    return $this->createImageBlock($imgNode, $hasText ? $textContent : null);
                }

                // Regular paragraph
                return $this->createParagraphBlock($element);

            case 'h1':
            case 'h2':
            case 'h3':
            case 'h4':
            case 'h5':
            case 'h6':
                return $this->createHeadingBlock($element);

            case 'ul':
            case 'ol':
                return $this->createListBlock($element);

            case 'blockquote':
                return $this->createQuoteBlock($element);

            case 'iframe':
                return $this->createIframeBlock($element);

            case 'div':
            case 'section':
                // Check for video containers
                return $this->createHtmlBlock($element);

            default:
                return null;
        }
    }

    /**
     * Create an image block from img element
     */
    private function createImageBlock(DOMElement $img, ?string $caption = null): string
    {
        $src = $img->getAttribute('src');
        $alt = $img->getAttribute('alt');
        $class = $img->getAttribute('class');
        
        // Extract image ID from wp-image-{id} class if present
        $imageId = null;
        if (preg_match('/wp-image-(\d+)/', $class, $matches)) {
            $imageId = intval($matches[1]);
        }
        
        // Use provided caption or alt text as fallback
        $captionText = $caption ?? $alt;
        $captionHtml = '';
        
        if ($captionText) {
            $captionHtml = sprintf(
                "<figcaption class=\"wp-element-caption\">%s</figcaption>",
                $captionText
            );
        }

        // Build block attributes
        $attributes = [
            'sizeSlug' => 'large'
        ];
        if ($imageId) {
            $attributes['id'] = $imageId;
        }
        $attributesJson = json_encode($attributes);

        return sprintf(
            "<!-- wp:image %s -->\n" .
            "<figure class=\"wp-block-image size-large\">" .
            "<img src=\"%s\" alt=\"%s\" class=\"%s\"/>%s</figure>\n" .
            "<!-- /wp:image -->",
            $attributesJson,
            $src,
            $alt,
            $class,
            $captionHtml
        );
    }

    /**
     * Create a paragraph block from p element
     */
    private function createParagraphBlock(DOMElement $p): string
    {
        $content = $this->getInnerHtml($p);
        $stripped = strip_tags($content);

        if (empty(trim($stripped))) {
            return '';
        }

        return sprintf(
            "<!-- wp:paragraph -->\n<p>%s</p>\n<!-- /wp:paragraph -->",
            $content
        );
    }

    /**
     * Create a heading block from h1-h6 element
     */
    private function createHeadingBlock(DOMElement $heading): string
    {
        $level = (int) substr($heading->tagName, 1);
        $content = $heading->textContent;

        return sprintf(
            "<!-- wp:heading {\"level\":%d} -->\n<%s>%s</%s>\n<!-- /wp:heading -->",
            $level,
            $heading->tagName,
            $content,
            $heading->tagName
        );
    }

    /**
     * Create a list block from ul/ol element
     */
    private function createListBlock(DOMElement $list): string
    {
        return sprintf(
            "<!-- wp:list {\"ordered\":%s} -->\n<%s>%s</%s>\n<!-- /wp:list -->",
            $list->tagName === 'ol' ? 'true' : 'false',
            $list->tagName,
            $this->getInnerHtml($list),
            $list->tagName
        );
    }

    /**
     * Create a quote block from blockquote element
     */
    private function createQuoteBlock(DOMElement $quote): string
    {
        return sprintf(
            "<!-- wp:quote -->\n<blockquote class=\"wp-block-quote\">%s</blockquote>\n<!-- /wp:quote -->",
            $this->getInnerHtml($quote)
        );
    }

    /**
     * Get inner HTML of a DOM element
     */
    private function getInnerHtml(DOMElement $element): string
    {
        $html = '';
        foreach ($element->childNodes as $child) {
            $html .= $element->ownerDocument->saveHTML($child);
        }
        return $html;
    }

    private function getOuterHtml(DOMElement $element): string
    {
        return $element->ownerDocument->saveHTML($element);
    }

    /**
     * Create a video block from video container element
     */
    private function createHtmlBlock(DOMElement $element): string
    {
        $html = $this->getOuterHtml($element);
        
        return sprintf(
            "<!-- wp:html -->\n%s\n<!-- /wp:html -->",
            $html
        );
    }

    /**
     * Create an iframe block
     */
    private function createIframeBlock(DOMElement $element): string
    {
        $src = $element->getAttribute('src');
        $width = $element->getAttribute('width') ?: '100%';
        $height = $element->getAttribute('height') ?: '480';
        $allowFullscreen = $element->hasAttribute('allowfullscreen') ? ' allowfullscreen' : '';
        $allow = $element->getAttribute('allow');
        $allowAttr = $allow ? sprintf(' allow="%s"', esc_attr($allow)) : '';
        $frameborder = $element->getAttribute('frameborder') ?: '0';
        
        

        // For non-video iframes, use standard format
        return sprintf(
            "<!-- wp:html -->\n<div class=\"wp-block-post-content\">" .
            "<iframe width=\"%s\" height=\"%s\" src=\"%s\" frameborder=\"%s\"%s%s></iframe>" .
            "</div>\n<!-- /wp:html -->",
            esc_attr($width),
            esc_attr($height),
            esc_url($src),
            esc_attr($frameborder),
            $allowAttr,
            $allowFullscreen
        );
    }
}