<?php

namespace T4S\Rewriter\Settings;

class SettingsManager {
    const OPTION_NAME = 't4s_rewriter_settings';
    
    private static $instance = null;
    private $settings;

    public static function getInstance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->settings = get_option(self::OPTION_NAME, $this->getDefaultSettings());
    }

    private function getDefaultSettings(): array {
        return [
            'api_key' => '',
            'language' => 'English',
            'download_images' => false,
            'last_usage_check' => '',
            'usage_data' => []
        ];
    }

    public function getApiKey(): string {
        return $this->settings['api_key'] ?? '';
    }

    public function getLanguage(): string {
        return $this->settings['language'] ?? 'en';
    }

    public function getLastUsageCheck(): string {
        return $this->settings['last_usage_check'] ?? '';
    }

    public function getUsageData(): array {
        return $this->settings['usage_data'] ?? [];
    }

    public function shouldDownloadImages(): bool {
        return $this->settings['download_images'] ?? false;
    }

    public function updateUsageData(array $usage): bool {
        return $this->updateSettings([
            'usage_data' => $usage,
            'last_usage_check' => current_time('mysql')
        ]);
    }

    public function shouldCheckUsage(): bool {
        $lastCheck = $this->getLastUsageCheck();
        if (empty($lastCheck)) {
            return true;
        }

        // Check usage every 5 minutes
        $nextCheck = strtotime($lastCheck . ' +5 minutes');
        return time() >= $nextCheck;
    }

    public function updateSettings(array $settings): bool {
        $newSettings = array_merge($this->settings, $settings);
        $updated = update_option(self::OPTION_NAME, $newSettings);
        if ($updated) {
            $this->settings = $newSettings;
        }
        
        return $updated;
    }

    public function validateApiKey(string $apiKey): bool {
        try {
            $apiClient = new \T4S\Rewriter\Api\ApiClient();
            return $apiClient->validateApiKey();
        } catch (\Exception $e) {
            return false;
        }
    }

    public function getAvailableLanguages(): array {
        return [
            'English' => 'English',
            'Vietnamese' => 'Vietnamese',
            'Spanish' => 'Spanish',
            'French' => 'French',
            'German' => 'German'
        ];
    }

    public function getAvailableWritingStyles(): array {
        return [
            'Auto' => 'Auto',
            'Professional' => 'Professional',
            'Casual' => 'Casual',
            'Academic' => 'Academic',
            'Creative' => 'Creative',
            'Technical' => 'Technical',
            'Journalistic' => 'Journalistic',
            'Informative' => 'Informative',
            'Conversational' => 'Conversational',
        ];
    }
}