<?php

namespace T4S\Rewriter;

use T4S\Rewriter\Admin\WelcomePage;
use T4S\Rewriter\Admin\SettingsPage;
use T4S\Rewriter\Admin\JobsPage;
use T4S\Rewriter\Database\Installer;

class Plugin {
    private static $instance = null;
    private $welcomePage;
    private $settingsPage;
    private $jobsPage;
    private $cronHandler;

    public static function getInstance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->welcomePage = new WelcomePage();
        $this->settingsPage = new SettingsPage();
        $this->jobsPage = new JobsPage();
        $this->cronHandler = new \T4S\Rewriter\Cron\CronHandler();
    }

    public function init() {
        // Initialize admin pages
        if (is_admin()) {
            $this->welcomePage->init();
            $this->settingsPage->init();
            $this->jobsPage->init();
        }
        
        // Initialize cron handler
        $this->cronHandler->init();
        
        add_action('init', [$this, 'onInit']);
    }

    public function onInit() {
        // Plugin initialization code
    }

    public static function activate() {
        try {
            // Install database tables
            Installer::install();
            
            // Set default options if not exist
            if (!get_option('t4s_rewriter_settings')) {
                update_option('t4s_rewriter_settings', [
                    'api_key' => '',
                    'language' => 'English',
                    'download_images' => false,
                    'last_usage_check' => '',
                    'usage_data' => []
                ]);
            }

            // Add custom cron interval
            add_filter('cron_schedules', function ($schedules) {
                $schedules['5min'] = [
                    'interval' => 300,
                    'display' => '5 minutes'
                ];
                $schedules['1min'] = [
                    'interval' => 60,
                    'display' => '1 minute'
                ];
                return $schedules;
            });

            // Schedule cron job
            if (!wp_next_scheduled(\T4S\Rewriter\Cron\CronHandler::RSS_FETCH_HOOK)) {
                wp_schedule_event(time(), '5min', \T4S\Rewriter\Cron\CronHandler::RSS_FETCH_HOOK);
            }
            if (!wp_next_scheduled(\T4S\Rewriter\Cron\CronHandler::REWRITE_HOOK)) {
                wp_schedule_event(time(), '1min', \T4S\Rewriter\Cron\CronHandler::REWRITE_HOOK);
            }

            // Flush rewrite rules
            flush_rewrite_rules();

        } catch (\Exception $e) {
            // Log the error
            error_log('T4S Rewriter activation failed: ' . $e->getMessage());
            
            // Clean up any partial activation
            self::cleanup_activation();
            
            // Deactivate the plugin
            deactivate_plugins(plugin_basename(T4S_REWRITER_PLUGIN_DIR . 't4s-rewriter.php'));
            
            // Throw error to show to admin
            wp_die('Could not activate T4S Rewriter plugin: ' . $e->getMessage());
        }
    }

    private static function cleanup_activation() {
        // Clean up tables if they were partially created
        global $wpdb;
        $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}t4s_rewriter_processed_urls");
        $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}t4s_rewriter_jobs");
        
        // Remove options
        delete_option('t4s_rewriter_settings');
        delete_option('t4s_rewriter_db_version');
    }

    public static function deactivate() {
        // Clear scheduled cron jobs
        \T4S\Rewriter\Cron\CronHandler::deactivate();
        
        // Clean up transients
        delete_transient('t4s_rewriter_admin_notices');
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }

    public static function uninstall() {
        // Clear scheduled cron jobs
        \T4S\Rewriter\Cron\CronHandler::deactivate();
        
        // Clean up database and options
        Installer::uninstall();
        
        // Clean up post meta
        global $wpdb;
        $wpdb->delete($wpdb->postmeta, ['meta_key' => '_t4s_original_url']);
        
        // Clear any remaining transients
        delete_transient('t4s_rewriter_admin_notices');
    }
}