<?php

namespace T4S\Rewriter\Job;

class JobManager {
    private static $instance = null;
    private $table;

    public static function getInstance(): self {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        global $wpdb;
        $this->table = $wpdb->prefix . 't4s_rewriter_jobs';
    }

    public function createJob(Job $job): int {
        global $wpdb;

        // Validate table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '$this->table'") !== $this->table) {
            throw new \Exception('Jobs table does not exist. Please deactivate and reactivate the plugin.');
        }

        // Validate required data
        if (empty($job->getRssUrl())) {
            throw new \Exception('RSS URL is required');
        }

        if (empty($job->getCategoryIds())) {
            throw new \Exception('At least one category is required');
        }

        if (empty($job->getWritingStyle())) {
            throw new \Exception('Writing style is required');
        }

        // Prepare data for insertion
        $data = $job->toArray();
        unset($data['id']); // Remove ID for insertion

        // Set timestamps
        $data['created_at'] = current_time('mysql');
        $data['updated_at'] = current_time('mysql');
        
        // Attempt insertion
        $inserted = $wpdb->insert(
            $this->table,
            $data,
            [
                '%s', // rss_url
                '%s', // category_ids (TEXT)
                '%s', // writing_style
                '%s', // post_status
                '%d', // is_active
                '%d', // articles_processed
                '%s', // last_run
                '%d', // daily_post_count
                '%d', // hourly_post_count
                '%d', // is_ascending
                '%s', // created_at
                '%s'  // updated_at
            ]
        );

        if ($inserted === false) {
            $error = $wpdb->last_error ?: 'Unknown database error';
            error_log('T4S Rewriter job creation failed: ' . $error);
            throw new \Exception('Failed to create job: ' . $error);
        }

        $insert_id = $wpdb->insert_id;
        if (!$insert_id) {
            throw new \Exception('Failed to get ID of created job');
        }

        return $insert_id;
    }

    public function updateJob(Job $job): bool {
        global $wpdb;

        $data = $job->toArray();
        $id = $data['id'];
        unset($data['id']); // Remove ID from update data
        unset($data['created_at']); // Don't update creation date
        
        // Add updated_at timestamp
        $data['updated_at'] = current_time('mysql');
        
        $updated = $wpdb->update(
            $this->table,
            $data,
            ['id' => $id],
            [
                '%s', // rss_url
                '%s', // category_ids (TEXT)
                '%s', // writing_style
                '%s', // post_status
                '%d', // is_active
                '%d', // articles_processed
                '%s', // last_run
                '%d', // daily_post_count
                '%d', // hourly_post_count
                '%d', // is_ascending
                '%s'  // updated_at
            ],
            ['%d'] // id format
        );

        return $updated !== false;
    }

    public function getJob(int $id): ?Job {
        global $wpdb;

        $row = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$this->table} WHERE id = %d", $id),
            ARRAY_A
        );

        return $row ? new Job($row) : null;
    }

    public function deleteJob(int $id): bool {
        global $wpdb;

        $deleted = $wpdb->delete(
            $this->table,
            ['id' => $id],
            ['%d']
        );

        return $deleted !== false;
    }

    public function getAllJobs(array $args = []): array {
        global $wpdb;

        $where = [];
        $values = [];
        
        if (isset($args['is_active'])) {
            $where[] = 'is_active = %d';
            $values[] = $args['is_active'];
        }

        // Handle category filtering with JSON search
        if (isset($args['category_id'])) {
            $where[] = 'JSON_CONTAINS(category_ids, %s)';
            $values[] = json_encode([$args['category_id']]);
        }

        $whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
        
        $query = "SELECT * FROM {$this->table} {$whereClause} ORDER BY created_at DESC";
        
        if (!empty($values)) {
            $query = $wpdb->prepare($query, ...$values);
        }

        $rows = $wpdb->get_results($query, ARRAY_A);
        
        return array_map(function($row) {
            return new Job($row);
        }, $rows ?: []);
    }

    public function toggleJobStatus(int $id, bool $active): bool {
        global $wpdb;

        $updated = $wpdb->update(
            $this->table,
            ['is_active' => $active],
            ['id' => $id],
            ['%d'],
            ['%d']
        );

        return $updated !== false;
    }

    public function validateRssUrl(string $url): bool {
        // Basic URL validation
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            return false;
        }

        // TODO: Implement more thorough RSS validation when needed
        return true;
    }
}