<?php

namespace T4S\Rewriter\Database;

class Installer {
    public static function install() {
        self::createTables();
        self::updateTables();
    }

    public static function createTables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $jobs_table = $wpdb->prefix . 't4s_rewriter_jobs';
        $processed_urls_table = $wpdb->prefix . 't4s_rewriter_processed_urls';
        
        $sql = [
            // Jobs table
            "CREATE TABLE IF NOT EXISTS $jobs_table (
                id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                rss_url text NOT NULL,
                category_ids text NOT NULL,
                writing_style varchar(255) NOT NULL,
                post_status varchar(20) NOT NULL DEFAULT 'draft',
                is_active tinyint(1) NOT NULL DEFAULT 1,
                articles_processed int(11) DEFAULT 0,
                last_run datetime DEFAULT NULL,
                daily_post_count int(11) DEFAULT 0,
                hourly_post_count int(11) DEFAULT 0,
                is_ascending tinyint(1) NOT NULL DEFAULT 0,
                require_featured_image tinyint(1) NOT NULL DEFAULT 1,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY  (id)
            ) $charset_collate;",

            // Processed URLs table
            "CREATE TABLE IF NOT EXISTS $processed_urls_table (
                id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                job_id bigint(20) unsigned NOT NULL,
                url_hash varchar(64) NOT NULL,
                url text NOT NULL,
                article_id bigint(20) DEFAULT NULL,
                status varchar(20) NOT NULL DEFAULT 'pending',
                attempt_count int(11) DEFAULT 0,
                error_message text DEFAULT NULL,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                processed_at datetime DEFAULT NULL,
                PRIMARY KEY  (id),
                UNIQUE KEY url_hash (url_hash),
                KEY job_id (job_id),
                KEY status (status),
                CONSTRAINT fk_job_id FOREIGN KEY (job_id) 
                    REFERENCES $jobs_table(id) ON DELETE CASCADE
            ) $charset_collate;"
        ];

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        // Run the creation/update
        foreach ($sql as $query) {
            $result = dbDelta($query);
            
            if (empty($result)) {
                // If dbDelta failed, try direct query
                $wpdb->query($query);
            }
        }

        // Verify tables exist
        $tables_missing = false;
        foreach ([$jobs_table, $processed_urls_table] as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '$table'") !== $table) {
                $tables_missing = true;
                error_log("T4S Rewriter: Failed to create table: $table");
            }
        }

        if ($tables_missing) {
            throw new \Exception('Failed to create required database tables. Check error log for details.');
        }

        // Update plugin version
        update_option('t4s_rewriter_db_version', '1.1.0');
    }

    private static function beforeUpdateTables() {
        global $wpdb;
        $jobs_table = $wpdb->prefix . 't4s_rewriter_jobs';
        $categoryBackup = [];
        
        // Check if category_id column exists
        $columns = $wpdb->get_results("SHOW COLUMNS FROM $jobs_table");
        $existing_columns = array_map(function($column) {
            return $column->Field;
        }, $columns);

        if (in_array('category_id', $existing_columns)) {
            // Backup category_id data
            $jobs = $wpdb->get_results("SELECT id, category_id FROM $jobs_table");
            foreach ($jobs as $job) {
                $categoryBackup[$job->id] = $job->category_id;
            }
        }

        return $categoryBackup;
    }

    private static function afterUpdateTables($categoryBackup) {
        global $wpdb;
        $jobs_table = $wpdb->prefix . 't4s_rewriter_jobs';

        // Check if we have backup data and category_ids column exists
        if (!empty($categoryBackup)) {
            $columns = $wpdb->get_results("SHOW COLUMNS FROM $jobs_table");
            $existing_columns = array_map(function($column) {
                return $column->Field;
            }, $columns);

            if (in_array('category_ids', $existing_columns)) {
                // Restore data to category_ids
                foreach ($categoryBackup as $id => $category_id) {
                    $category_ids = json_encode([$category_id]);
                    $wpdb->query($wpdb->prepare(
                        "UPDATE $jobs_table SET category_ids = %s WHERE id = %d",
                        $category_ids,
                        $id
                    ));
                }
            }
        }
    }

    public static function updateTables() {
        global $wpdb;
        
        // Backup existing data
        $categoryBackup = self::beforeUpdateTables();

        // Define table updates
        $updates = [
            // Jobs table updates
            $wpdb->prefix . 't4s_rewriter_jobs' => [
                'add' => [
                    'daily_post_count' => "ADD COLUMN daily_post_count int(11) DEFAULT 0",
                    'hourly_post_count' => "ADD COLUMN hourly_post_count int(11) DEFAULT 0",
                    'is_ascending' => "ADD COLUMN is_ascending tinyint(1) NOT NULL DEFAULT 0",
                    'category_ids' => "ADD COLUMN category_ids text NOT NULL",
                    'require_featured_image' => "ADD COLUMN require_featured_image tinyint(1) NOT NULL DEFAULT 1"
                ],
                'modify' => [
                    // Example: 'column_name' => "MODIFY COLUMN column_name new_definition"
                ],
                'remove' => [
                    'category_id' => "DROP COLUMN category_id"
                ]
            ]
        ];

        // Apply updates for each table
        foreach ($updates as $table => $modifications) {
            // Get existing columns
            $columns = $wpdb->get_results("SHOW COLUMNS FROM $table");
            $existing_columns = array_map(function($column) {
                return $column->Field;
            }, $columns);

            // Process column additions
            if (!empty($modifications['add'])) {
                foreach ($modifications['add'] as $column => $sql) {
                    if (!in_array($column, $existing_columns)) {
                        $wpdb->query("ALTER TABLE $table $sql");
                    }
                }
            }

            // Process column modifications
            if (!empty($modifications['modify'])) {
                foreach ($modifications['modify'] as $column => $sql) {
                    if (in_array($column, $existing_columns)) {
                        $wpdb->query("ALTER TABLE $table $sql");
                    }
                }
            }

            // Process column removals
            if (!empty($modifications['remove'])) {
                foreach ($modifications['remove'] as $column => $sql) {
                    if (in_array($column, $existing_columns)) {
                        $wpdb->query("ALTER TABLE $table $sql");
                    }
                }
            }
        }

        // Restore data to new column
        self::afterUpdateTables($categoryBackup);
    }

    public static function uninstall() {
        global $wpdb;
        
        // Drop tables
        $tables = [
            $wpdb->prefix . 't4s_rewriter_processed_urls',
            $wpdb->prefix . 't4s_rewriter_jobs'
        ];

        foreach ($tables as $table) {
            $wpdb->query("DROP TABLE IF EXISTS $table");
        }
        
        // Remove options
        delete_option('t4s_rewriter_db_version');
        delete_option('t4s_rewriter_settings');

        // Clean up post meta
        $wpdb->delete($wpdb->postmeta, ['meta_key' => '_t4s_original_url']);
    }
}