<?php

namespace T4S\Rewriter\Cron;

class VideoHandler
{
    /**
     * Generate HTML code for videos based on type and platform
     * @param array $videos Array of video objects with type, url and optional platform
     * @return string Generated HTML code
     */
    public function processVideos(array $videos): string
    {
        $html = '';
        foreach ($videos as $video) {
            $html .= $this->generateVideoHtml($video) . "\n\n";
        }
        return $html;
    }

    /**
     * Generate HTML for a single video
     * @param array $video Video object with type, url and optional platform
     * @return string Generated HTML code
     */
    private function generateVideoHtml(array $video): string
    {
        if (!isset($video['type']) || !isset($video['url'])) {
            error_log('T4S Rewriter: Invalid video format - missing type or url');
            return '';
        }

        switch ($video['type']) {
            case 'video-html5':
                return $this->generateHtml5Video($video['url']);

            case 'hls':
                return $this->generateHlsVideo($video['url']);

            case 'iframe':
                return $this->generateIframeVideo($video);

            default:
                error_log("T4S Rewriter: Unknown video type: {$video['type']}");
                return '';
        }
    }

    /**
     * Generate HTML5 video player
     */
    private function generateHtml5Video(string $url): string
    {
        return sprintf(
            '<div class="wp-block-columns items-justified-center">
                <video controls width="100%" playsinline>
                    <source src="%s" type="video/mp4">
                    Your browser does not support the video tag.
                </video>
            </div>',
            esc_url($url)
        );
    }

    /**
     * Generate HLS video player with default HLS.js implementation
     */
    private function generateHlsVideo(string $url): string
    {
        $uniqueId = 'hls-' . substr(uniqid(), -6);
        return sprintf(
            '<div class="wp-block-columns items-justified-center">
                <video id="%s" controls width="100%%" playsinline></video>
                <script src="https://cdn.jsdelivr.net/npm/hls.js@latest"></script>
                <script>
                    (function() {
                        const video = document.getElementById("%s");
                        if (Hls.isSupported()) {
                            const hls = new Hls();
                            hls.loadSource("%s");
                            hls.attachMedia(video);
                        }
                        else if (video.canPlayType("application/vnd.apple.mpegurl")) {
                            video.src = "%s";
                        }
                    })();
                </script>
            </div>',
            esc_attr($uniqueId),
            esc_attr($uniqueId),
            esc_url($url),
            esc_url($url)
        );
    }

    /**
     * Generate iframe embed based on platform
     */
    private function generateIframeVideo(array $video): string
    {
        $url = $video['url'];
        $platform = $video['platform'] ?? $this->detectPlatform($url);

        switch ($platform) {
            case 'youtube':
                return $this->generateYouTubeEmbed($url);
            
            case 'vimeo':
                return $this->generateVimeoEmbed($url);
            
            case 'facebook':
                return $this->generateFacebookEmbed($url);

            case 'instagram':
                return $this->generateInstagramEmbed($url);
            
            default:
                // Generic iframe if platform not specified or unknown
                return sprintf(
                    '<div class="wp-block-columns items-justified-center">
                        <iframe width="100%%" height="480" src="%s" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe>
                    </div>',
                    esc_url($url)
                );
        }
    }

    /**
     * Detect platform from URL if not specified
     */
    private function detectPlatform(string $url): ?string
    {
        if (strpos($url, 'youtube.com') !== false || strpos($url, 'youtu.be') !== false) {
            return 'youtube';
        }
        if (strpos($url, 'vimeo.com') !== false) {
            return 'vimeo';
        }
        if (strpos($url, 'facebook.com') !== false) {
            return 'facebook';
        }
        if (strpos($url, 'instagram.com') !== false) {
            return 'instagram';
        }
        return null;
    }

    /**
     * Generate Instagram embed
     */
    private function generateInstagramEmbed(string $url): string
    {
        // Extract post ID from URL
        preg_match('/instagram\.com\/(?:p|reel)\/([^\/\?\&]+)/', $url, $matches);
        
        if (!isset($matches[1])) {
            error_log("T4S Rewriter: Invalid Instagram URL: {$url}");
            return '';
        }

        $postId = $matches[1];
        
        return sprintf(
            '<div class="wp-block-columns items-justified-center">
                <blockquote class="instagram-media" data-instgrm-permalink="%s" data-instgrm-version="14">
                    <a href="%s">View this post on Instagram</a>
                </blockquote>
                <script async src="//www.instagram.com/embed.js"></script>
            </div>',
            esc_url($url),
            esc_url($url)
        );
    }

    /**
     * Generate YouTube embed
     */
    private function generateYouTubeEmbed(string $url): string
    {
        // Extract video ID
        preg_match('/(?:youtube\.com\/(?:[^\/\n\s]+\/\S+\/|(?:v|e(?:mbed)?)\/|\S*?[?&]v=)|youtu\.be\/)([a-zA-Z0-9_-]{11})/', $url, $matches);
        
        if (!isset($matches[1])) {
            error_log("T4S Rewriter: Invalid YouTube URL: {$url}");
            return '';
        }

        return sprintf(
            '<div class="wp-block-columns items-justified-center">
                <iframe width="100%%" height="480" src="https://www.youtube.com/embed/%s" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe>
            </div>',
            esc_attr($matches[1])
        );
    }

    /**
     * Generate Vimeo embed
     */
    private function generateVimeoEmbed(string $url): string
    {
        // Extract video ID
        preg_match('/vimeo\.com\/(?:channels\/(?:\w+\/)?|groups\/([^\/]*)\/videos\/|album\/(\d+)\/video\/|)(\d+)(?:$|\/|\?)/', $url, $matches);
        
        if (!isset($matches[3])) {
            error_log("T4S Rewriter: Invalid Vimeo URL: {$url}");
            return '';
        }

        return sprintf(
            '<div class="wp-block-columns items-justified-center">
                <iframe width="100%%" height="480" src="https://player.vimeo.com/video/%s" frameborder="0" allow="autoplay; fullscreen; picture-in-picture" allowfullscreen></iframe>
            </div>',
            esc_attr($matches[3])
        );
    }

    /**
     * Generate Facebook embed
     */
    private function generateFacebookEmbed(string $url): string
    {
        return sprintf(
            '<div class="wp-block-columns items-justified-center">
                <div class="fb-video" data-href="%s" data-width="100%%" data-show-text="false">
                    <blockquote cite="%s" class="fb-xfbml-parse-ignore"></blockquote>
                </div>
                <script async defer src="https://connect.facebook.net/en_US/sdk.js#xfbml=1&version=v3.2"></script>
            </div>',
            esc_url($url),
            esc_url($url)
        );
    }
}