<?php

namespace T4S\Rewriter\Cron;

class ImageHandler
{
    // Allowed image types
    private const ALLOWED_IMAGE_TYPES = [
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
        'gif' => 'image/gif',
        'webp' => 'image/webp'
    ];

    /**
     * Validate image URL and get file info
     * @return array|null [file_name, mime_type] or null if invalid
     */
    private function validateImageUrl(string $url): ?array
    {
        // Parse URL and get path
        $parsed_url = parse_url($url);
        if (!isset($parsed_url['path'])) {
            error_log("T4S Rewriter: Invalid URL format: {$url}");
            return null;
        }

        // Get filename from path
        $file_name = basename($parsed_url['path']);
        
        // Get extension
        $extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        $mime_type = null;

        // If no extension or unknown extension, try to determine from content type header
        if (empty($extension) || !isset(self::ALLOWED_IMAGE_TYPES[$extension])) {
            // Get headers from URL
            $headers = get_headers($url, 1);
            
            if ($headers && isset($headers['Content-Type'])) {
                $content_type = is_array($headers['Content-Type']) 
                    ? $headers['Content-Type'][0] 
                    : $headers['Content-Type'];
                
                // Find extension by content type
                $extension = array_search($content_type, self::ALLOWED_IMAGE_TYPES);
                
                if ($extension !== false) {
                    $mime_type = $content_type;
                    // If no extension in filename, add it
                    if (empty(pathinfo($file_name, PATHINFO_EXTENSION))) {
                        $file_name .= '.' . $extension;
                    }
                } else {
                    // If content type is image/* but not in our allowed types, default to jpg
                    if (strpos($content_type, 'image/') === 0) {
                        $extension = 'jpg';
                        $mime_type = 'image/jpeg';
                        $file_name .= '.jpg';
                    } else {
                        error_log("T4S Rewriter: Unsupported content type: {$content_type} for URL: {$url}");
                        return null;
                    }
                }
            } else {
                // If can't determine content type, default to jpg
                error_log("T4S Rewriter: Could not determine content type for URL: {$url}, defaulting to jpg");
                $extension = 'jpg';
                $mime_type = 'image/jpeg';
                $file_name .= '.jpg';
            }
        } else {
            $mime_type = self::ALLOWED_IMAGE_TYPES[$extension];
        }

        // Clean filename and ensure it has correct extension
        $clean_file_name = preg_replace('/[^a-zA-Z0-9\-\_\.]/', '', $file_name);
        if (!preg_match('/\.' . $extension . '$/', $clean_file_name)) {
            $clean_file_name .= '.' . $extension;
        }

        // Add unique identifier to filename to prevent conflicts
        $name_parts = pathinfo($clean_file_name);
        $unique_name = sprintf(
            '%s-%s.%s',
            $name_parts['filename'],
            substr(uniqid(), -6),
            $name_parts['extension']
        );

        return [
            'file_name' => $unique_name,
            'mime_type' => $mime_type
        ];
    }

    /**
     * Download and add image to media library
     * @param string $url Image URL to download
     * @param string $alt_text ALT text for the image
     * @return int|null Attachment ID or null on failure
     */
    private function downloadAndAddToMediaLibrary(string $url, string $alt_text = ''): ?int
    {
        $image_info = $this->validateImageUrl($url);
        if (!$image_info) {
            return null;
        }

        try {
            $tmp_file = download_url($url);
            
            if (is_wp_error($tmp_file)) {
                error_log('T4S Rewriter: Error downloading image: ' . $tmp_file->get_error_message());
                return null;
            }

            // Prepare file array with proper mime type
            $file_array = [
                'name' => $image_info['file_name'],
                'tmp_name' => $tmp_file,
                'type' => $image_info['mime_type']
            ];

            // Insert into media library with description
            $attachment_id = media_handle_sideload($file_array, 0, $alt_text);

            if (is_wp_error($attachment_id)) {
                error_log('T4S Rewriter: Error adding image to media library: ' . $attachment_id->get_error_message());
                unlink($tmp_file);
                return null;
            }

            return $attachment_id;

        } catch (\Exception $e) {
            error_log('T4S Rewriter: Error processing image: ' . $e->getMessage());
            if (isset($tmp_file) && file_exists($tmp_file)) {
                unlink($tmp_file);
            }
            return null;
        }
    }

    /**
     * Get first image from content and set as featured image
     */
    public function setFeaturedImageFromContent(string $content, array $images = [], string $featured_image = null): ?int
    {
        // Match old_url from images array with featured_image
        if ($featured_image) {
            foreach ($images as $image) {
                if ($image['old_url'] === $featured_image) {
                    \error_log("T4S Rewriter: Found featured image in downloaded images: {$featured_image}");
                    return $image['attachment_id'];
                }
            }
        }
        // Find first image in markdown content
        if (!preg_match('/!\[([^\]]*)\]\(([^\)]+)\)/', $content, $match)) {
            // If no image found in content, try to get first image from already downloaded images
            if (!empty($images)) {
                return $images[0]['attachment_id']; // Return first attachment_id
            }
            
            // No images found at all
            return null;
        }

        $image_url = $match[2];
        $alt_text = $match[1];

        // Check if we already downloaded this image
        foreach ($images as $image) {
            if ($image['url'] === $image_url) {
                return $image['attachment_id']; // Return attachment_id
            }
        }
        \error_log("T4S Rewriter: Image not found in downloaded images: {$image_url}");
        // If not found, download and add to media library with ALT text
        return $this->downloadAndAddToMediaLibrary($image_url, $alt_text);
    }

    /**
     * Process markdown images - download and replace URLs
     * @return array [string $content, array $image_mappings]
     */
    public function processMarkdownImages(string $content): array
    {
        // Match markdown image syntax ![alt](url)
        preg_match_all('/!\[([^\]]*)\]\(([^\)]+)\)/', $content, $matches);
        
        if (empty($matches[2])) {
            return [$content, []];
        }

        $image_urls = $matches[2];
        $alt_texts = $matches[1];
        $images = [];

        foreach ($image_urls as $index => $url) {
            $alt_text = $alt_texts[$index] ?? '';
            $attachment_id = $this->downloadAndAddToMediaLibrary($url, $alt_text);
            if ($attachment_id) {
                $new_url = wp_get_attachment_url($attachment_id);
                $images[] = [
                    'url' => $new_url,
                    'old_url' => $url,
                    'attachment_id' => $attachment_id
                ];
                
                // Build regex to match the full markdown image syntax
                $escaped_url = preg_quote($url, '/');
                $pattern = '/!\[([^\]]*)\]\(' . $escaped_url . '\)/';
                
                // Build replacement with wp-image class
                $replacement = '![${1}](' . $new_url . '){.wp-image-' . $attachment_id . '}';
                
                $content = preg_replace($pattern, $replacement, $content);
            }
        }

        return [$content, $images];
    }

    /**
     * Attach downloaded images to a post
     * @param int $post_id The post ID to attach images to
     * @param array $images Array of [url, attachment_id] pairs
     */
    public function attachImagesToPost(int $post_id, array $images): void
    {
        if (empty($images)) {
            return;
        }

        foreach ($images as $image) {
            $attachment_id = $image['attachment_id'];
            wp_update_post([
                'ID' => $attachment_id,
                'post_parent' => $post_id
            ]);
        }
    }
}