<?php

namespace T4S\Rewriter\Cron;

use T4S\Rewriter\Settings\SettingsManager;

class CronHandler
{
    const RSS_FETCH_HOOK = 't4s_rewriter_fetch_rss';
    const REWRITE_HOOK = 't4s_rewriter_process_article';
    const CLEANUP_HOOK = 't4s_rewriter_cleanup_database';

    private $rssFeedProcessor;
    private $articleProcessor;
    private $databaseCleaner;
    private $settingsManager;

    public function __construct()
    {
        $this->rssFeedProcessor = new RssFeedProcessor();
        $this->articleProcessor = new ArticleProcessor();
        $this->databaseCleaner = new DatabaseCleaner();
        $this->settingsManager = SettingsManager::getInstance();
    }

    public function init()
    {
        // Register cron hooks
        add_action(self::RSS_FETCH_HOOK, [$this->rssFeedProcessor, 'processJobs']);
        add_action(self::REWRITE_HOOK, [$this->articleProcessor, 'processOneArticle']);
        add_action(self::CLEANUP_HOOK, [$this->databaseCleaner, 'cleanup']);
        
        add_filter('cron_schedules', function ($schedules) {
            $schedules['5min'] = [
                'interval' => 300,
                'display' => '5 minutes'
            ];
            $schedules['1min'] = [
                'interval' => 60,
                'display' => '1 minute'
            ];
            return $schedules;
        });

        // Schedule crons if not already scheduled
        if (!wp_next_scheduled(self::RSS_FETCH_HOOK)) {
            wp_schedule_event(time(), '5min', self::RSS_FETCH_HOOK);
        }

        if (!wp_next_scheduled(self::REWRITE_HOOK)) {
            wp_schedule_event(time(), '1min', self::REWRITE_HOOK);
        }

        // Schedule daily cleanup at midnight
        if (!wp_next_scheduled(self::CLEANUP_HOOK)) {
            // Get timestamp for next midnight in site's timezone
            $timezone = get_option('timezone_string') ?: 'UTC';
            $date = new \DateTime('tomorrow', new \DateTimeZone($timezone));
            $date->setTime(0, 0);
            
            wp_schedule_event($date->getTimestamp(), 'daily', self::CLEANUP_HOOK);
        }
    }

    public static function deactivate()
    {
        wp_clear_scheduled_hook(self::RSS_FETCH_HOOK);
        wp_clear_scheduled_hook(self::REWRITE_HOOK);
        wp_clear_scheduled_hook(self::CLEANUP_HOOK);
    }

    /**
     * Process one pending article (can be specific or next in queue)
     */
    public function processOneArticle($specific_url_id = null)
    {
        return $this->articleProcessor->processOneArticle($specific_url_id);
    }

    /**
     * Process RSS feeds for given jobs
     * @param array $jobs List of jobs to process
     */
    public function processJobs(array $jobs = [])
    {
        return $this->rssFeedProcessor->processJobs($jobs);
    }
}
