<?php

namespace T4S\Rewriter\Cron;

use T4S\Rewriter\Api\ApiClient;
use T4S\Rewriter\Settings\SettingsManager;
use T4S\Rewriter\Util\BlockConverter;
use \ParsedownExtra;

class ArticleProcessor
{
    private $apiClient;
    private $settingsManager;
    private $imageHandler;
    private $videoHandler;
    private $blockConverter;

    public function __construct()
    {
        $this->apiClient = new ApiClient();
        $this->settingsManager = SettingsManager::getInstance();
        $this->imageHandler = new ImageHandler();
        $this->videoHandler = new VideoHandler();
        $this->blockConverter = new BlockConverter();
    }

    private function canProcessMore(): bool
    {
        try {
            $usage = $this->apiClient->getUsage();
            return ($usage['remaining'] ?? 0) > 0;
        } catch (\Exception $e) {
            error_log('T4S Rewriter: Error checking API usage: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if we've hit the daily post limit for the specified job
     */
    private function isDailyLimitReached(int $job_id, int $daily_limit): bool
    {
        if ($daily_limit === 0) {
            return false; // 0 means unlimited
        }

        global $wpdb;
        $today_start = date('Y-m-d 00:00:00');
        $today_end = date('Y-m-d 23:59:59');

        $completed_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) 
            FROM {$wpdb->prefix}t4s_rewriter_processed_urls 
            WHERE job_id = %d 
            AND status = 'completed' 
            AND processed_at BETWEEN %s AND %s",
            $job_id,
            $today_start,
            $today_end
        ));

        return (int)$completed_count >= $daily_limit;
    }

    /**
     * Check if we've hit the hourly post limit for the specified job
     */
    private function isHourlyLimitReached(int $job_id, int $hourly_limit): bool
    {
        if ($hourly_limit === 0) {
            return false; // 0 means unlimited
        }

        global $wpdb;
        $current_hour = date('Y-m-d H:00:00');
        $next_hour = date('Y-m-d H:59:59');

        $completed_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) 
            FROM {$wpdb->prefix}t4s_rewriter_processed_urls 
            WHERE job_id = %d 
            AND status = 'completed' 
            AND processed_at BETWEEN %s AND %s",
            $job_id,
            $current_hour,
            $next_hour
        ));

        return (int)$completed_count >= $hourly_limit;
    }

    /**
     * Process one pending article (can be specific or next in queue)
     */
    public function processOneArticle($specific_url_id = null)
    {
        global $wpdb;

        if (!$this->canProcessMore()) {
            error_log('T4S Rewriter: API usage limit reached, skipping article processing');
            return false;
        }

        // Get article to process
        if ($specific_url_id) {
            $article = $wpdb->get_row($wpdb->prepare(
                "SELECT p.*, j.writing_style, j.post_status, j.category_ids,
                        j.daily_post_count, j.hourly_post_count, j.require_featured_image
                FROM {$wpdb->prefix}t4s_rewriter_processed_urls p
                JOIN {$wpdb->prefix}t4s_rewriter_jobs j ON p.job_id = j.id
                WHERE p.id = %d AND p.status != 'completed'",
                $specific_url_id
            ));
        } else {
            // First get jobs that haven't reached their limits
            $jobs_query = "SELECT j.id, j.daily_post_count, j.hourly_post_count, j.is_ascending 
                          FROM {$wpdb->prefix}t4s_rewriter_jobs j 
                          WHERE j.is_active = 1";
            
            $available_jobs = [];
            $jobs = $wpdb->get_results($jobs_query);
            
            foreach ($jobs as $job) {
                if (!$this->isDailyLimitReached($job->id, $job->daily_post_count) && 
                    !$this->isHourlyLimitReached($job->id, $job->hourly_post_count)) {
                    $available_jobs[] = $job;
                }
            }

            if (empty($available_jobs)) {
                error_log('T4S Rewriter: All active jobs have reached their limits');
                return false;
            }

            // Randomly select one job from available jobs
            $selected_job = $available_jobs[array_rand($available_jobs)];
            
            // Get next pending article for the selected job
            $article = $wpdb->get_row($wpdb->prepare(
                "SELECT p.*, j.writing_style, j.post_status, j.category_ids,
                        j.daily_post_count, j.hourly_post_count, j.is_ascending,
                        j.require_featured_image
                 FROM {$wpdb->prefix}t4s_rewriter_processed_urls p
                 JOIN {$wpdb->prefix}t4s_rewriter_jobs j ON p.job_id = j.id
                 WHERE p.status = 'pending'
                 AND j.is_active = 1
                 AND j.id = %d
                 ORDER BY p.id " . ($selected_job->is_ascending ? 'ASC' : 'DESC') . "
                 LIMIT 1",
                $selected_job->id
            ));
        }

        if (!$article) {
            return false;
        }

        // Double check limits for the specific article's job
        if (!$specific_url_id && (
            $this->isDailyLimitReached($article->job_id, $article->daily_post_count) ||
            $this->isHourlyLimitReached($article->job_id, $article->hourly_post_count)
        )) {
            error_log("T4S Rewriter: Job {$article->job_id} has reached its processing limits");
            return false;
        }

        try {
            // Call rewrite API
            $result = $this->apiClient->rewriteArticle(
                $article->url,
                $article->writing_style,
                $this->settingsManager->getLanguage()
            );

            $content = $result['content'];
            $featuredImage = isset($result['featured_image']) ? $result['featured_image'] : null;
            if($article->require_featured_image && !$featuredImage){
                throw new \Exception('No featured image found');
            }
            $images = [];

            // Process images if enabled
            if ($this->settingsManager->shouldDownloadImages()) {
                list($content, $images) = $this->imageHandler->processMarkdownImages($content);
            }

            // Process videos if available
            $video_html = '';
            if (!empty($result['videos'])) {
                //\error_log("Videos: " . json_encode($result['videos']));
                $video_html = $this->videoHandler->processVideos($result['videos']);
                //\error_log("Video HTML: " . $video_html);
            }

            // Convert markdown content to HTML with extra features
            $parsedown = new ParsedownExtra();
            $html_content = $video_html . $parsedown->text($content);

            // Convert content based on editor support
            $final_content = '';
            if (function_exists('use_block_editor_for_post')) {
                // Convert HTML to blocks
                $final_content = $this->blockConverter->fromHtml($html_content);
            } else {
                // Use HTML for classic editor
                $final_content = $html_content;
            }

            // Safety check - if block conversion failed, use HTML content
            if (empty($final_content)) {
                $final_content = $html_content;
            }

            // Get featured image from first image in content
            $featured_image_id = $this->imageHandler->setFeaturedImageFromContent($content, $images, $featuredImage);

            // Get category IDs from JSON
            $category_ids = json_decode($article->category_ids, true) ?: [];

            // Create WordPress post
            $post_data = [
                'post_title' => $result['title'],
                'post_content' => $final_content,
                'post_excerpt' => $result['description'],
                'post_status' => $article->post_status,
                'post_category' => $category_ids, // Set multiple categories
                'post_author' => 1,
                'tags_input' => $result['tags']
            ];

            $post_id = \wp_insert_post($post_data);
            
            // Set featured image if one was found
            if ($featured_image_id && !is_wp_error($post_id)) {
                \set_post_thumbnail($post_id, $featured_image_id);
            }

            if (is_wp_error($post_id)) {
                throw new \Exception('Failed to create post: ' . $post_id->get_error_message());
            }

            // Store original URL as post meta
            \add_post_meta($post_id, '_t4s_original_url', $article->url, true);

            // Attach downloaded images to the post
            if (!empty($images)) {
                $this->imageHandler->attachImagesToPost($post_id, $images);
            }

            // Update processed URL status
            $wpdb->update(
                $wpdb->prefix . 't4s_rewriter_processed_urls',
                [
                    'status' => 'completed',
                    'article_id' => $post_id,
                    'processed_at' => current_time('mysql')
                ],
                ['id' => $article->id],
                ['%s', '%d', '%s'],
                ['%d']
            );

            // Update job articles count
            $wpdb->query($wpdb->prepare(
                "UPDATE {$wpdb->prefix}t4s_rewriter_jobs 
                 SET articles_processed = articles_processed + 1 
                 WHERE id = %d",
                $article->job_id
            ));
            \error_log("T4S Rewriter: Rewritten article {$post_id} with title: {$result['title']}");
            return true;
        } catch (\Exception $e) {
            error_log('T4S Rewriter: Error processing article: ' . $e->getMessage());
            if (preg_match('/Rate limit|tunnel|abort|OpenRouter API/', $e->getMessage())) {
                return false;
            }
            // Update URL status to failed
            $wpdb->update(
                $wpdb->prefix . 't4s_rewriter_processed_urls',
                [
                    'status' => 'failed',
                    'error_message' => $e->getMessage(),
                    'processed_at' => current_time('mysql')
                ],
                ['id' => $article->id],
                ['%s', '%s', '%s'],
                ['%d']
            );

            return false;
        }
    }
}