<?php

namespace T4S\Rewriter\Api;

use T4S\Rewriter\Settings\SettingsManager;

class ApiClient {
    //const API_BASE_URL = 'http://127.0.0.1:8000';
    const API_BASE_URL = 'https://api-rewriter.traffic4seo.com';
    private $apiKey;
    private $settingsManager;

    public function __construct() {
        $this->settingsManager = SettingsManager::getInstance();
        $this->apiKey = $this->settingsManager->getApiKey();
    }

    private function request(string $endpoint, string $method = 'GET', array $data = []): array {
        $url = self::API_BASE_URL . $endpoint;
        
        $args = [
            'headers' => [
                'x-api-key' => $this->apiKey,
                'Content-Type' => 'application/json',
            ],
            'method' => $method,
            'timeout' => 60,
        ];

        if (!empty($data) && $method !== 'GET') {
            $args['body'] = json_encode($data);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            throw new \Exception('API request failed: ' . $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new \Exception('Invalid JSON response from API');
        }

        if ($code === 429) {
            throw new \Exception($data['message'] ?? 'Rate limit exceeded');
        }

        if ($code >= 400) {
            throw new \Exception($data['error'] ?? $data['message'] ?? 'API request failed');
        }

        return $data;
    }

    public function getAccountInfo(): array {
        return $this->request('/auth/me');
    }

    public function getUsage(): array {
        $data = $this->request('/api/rewrite/usage');
        return [
            'used' => $data['current'] ?? 0,
            'limit' => $data['limit'] ?? 0,
            'remaining' => $data['remaining'] ?? 0,
            'nextReset' => $data['nextReset'] ?? '',
            'type' => $data['type'] ?? 'user'
        ];
    }

    public function rewriteArticle(string $url, string $writingStyle, string $language): array {
        $response = $this->request('/api/rewrite', 'POST', [
            'url' => $url,
            'writingStyle' => $writingStyle,
            'language' => $language
        ]);
        

        return [
            'title' => $response['title'] ?? '',
            'content' => $response['rewritten_content'] ?? '',
            'description' => $response['description'] ?? '',
            'tags' => $response['tags'] ?? '',
            'original_url' => $response['original_url'] ?? $url,
            'videos' => $response['videos'] ?? [],
            'usage' => $response['usage'] ?? []
        ];
    }

    public function validateApiKey(): bool {
        try {
            $this->getAccountInfo();
            return true;
        } catch (\Exception $e) {
            if (str_contains($e->getMessage(), 'Rate limit')) {
                // If it's a rate limit error, the API key is still valid
                return true;
            }
            return false;
        }
    }
}