<?php

namespace T4S\Rewriter\Admin\View;

use T4S\Rewriter\Job\Job;
use T4S\Rewriter\Job\JobManager;
use T4S\Rewriter\Settings\SettingsManager;

class JobsViewController {
    private $jobManager;
    private $settingsManager;
    private $items_per_page = 10;

    public function __construct(JobManager $jobManager, SettingsManager $settingsManager) {
        $this->jobManager = $jobManager;
        $this->settingsManager = $settingsManager;
    }

    public function renderPage() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $action = $_GET['action'] ?? 'list';
        $job_id = isset($_GET['job_id']) ? (int)$_GET['job_id'] : 0;

        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">
                <?php echo esc_html(get_admin_page_title()); ?>
            </h1>
            
            <?php if ($action === 'list'): ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=t4s-rewriter-jobs&action=new')); ?>" 
                   class="page-title-action">Add New</a>
            <?php endif; ?>

            <?php
            switch ($action) {
                case 'new':
                    $this->renderJobForm();
                    break;
                case 'edit':
                    $this->renderJobForm($job_id);
                    break;
                case 'view':
                    $this->renderJobDetails($job_id);
                    break;
                default:
                    $this->renderJobsList();
                    break;
            }
            ?>
        </div>
        <?php
    }

    private function renderJobsList() {
        $jobs = $this->jobManager->getAllJobs();
        require __DIR__ . '/../templates/jobs-list.php';
    }

    private function renderJobDetails($job_id) {
        global $wpdb;
        $job = $this->jobManager->getJob($job_id);
        
        if (!$job) {
            wp_die('Job not found');
        }

        // Get current page and status filter
        $current_page = isset($_GET['paged']) ? max(1, (int)$_GET['paged']) : 1;
        $status_filter = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
        $offset = ($current_page - 1) * $this->items_per_page;

        // Prepare WHERE clause
        $where_clause = "WHERE job_id = %d";
        $query_params = [$job_id];

        if ($status_filter) {
            $where_clause .= " AND status = %s";
            $query_params[] = $status_filter;
        }

        // Get total items count with filter
        $total_items = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}t4s_rewriter_processed_urls $where_clause",
                ...$query_params
            )
        );

        // Calculate total pages
        $total_pages = ceil($total_items / $this->items_per_page);

        // Add pagination params
        $query_params[] = $this->items_per_page;
        $query_params[] = $offset;

        // Get processed URLs for this job with pagination and filter
        $processed_urls = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}t4s_rewriter_processed_urls
                 $where_clause
                 ORDER BY id DESC
                 LIMIT %d OFFSET %d",
                ...$query_params
            )
        );

        // Get available statuses for dropdown
        $available_statuses = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT DISTINCT status FROM {$wpdb->prefix}t4s_rewriter_processed_urls WHERE job_id = %d",
                $job_id
            )
        );

        // Prepare pagination and filter data
        $pagination = [
            'current_page' => $current_page,
            'total_pages' => $total_pages,
            'total_items' => $total_items,
            'items_per_page' => $this->items_per_page,
            'status_filter' => $status_filter,
            'available_statuses' => $available_statuses
        ];

        require __DIR__ . '/../templates/job-details.php';
    }

    private function renderJobForm($job_id = 0) {
        $job = $job_id ? $this->jobManager->getJob($job_id) : new Job();
        $categories = get_categories(['hide_empty' => false]);
        $writing_styles = $this->settingsManager->getAvailableWritingStyles();
        
        require __DIR__ . '/../templates/job-form.php';
    }
}