<?php

namespace T4S\Rewriter\Admin;

use T4S\Rewriter\Settings\SettingsManager;

class SettingsPage {
    private $settingsManager;
    private $apiClient;

    public function __construct() {
        $this->settingsManager = SettingsManager::getInstance();
        $this->apiClient = new \T4S\Rewriter\Api\ApiClient();
    }

    public function init() {
        add_action('admin_menu', [$this, 'addSettingsPage']);
        add_action('admin_init', [$this, 'registerSettings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueueScripts']);
    }

    public function enqueueScripts($hook) {
        if ('t4s-rewriter_page_t4s-rewriter-settings' !== $hook) {
            return;
        }

        wp_enqueue_style(
            't4s-rewriter-admin',
            plugins_url('/assets/css/admin.css', dirname(dirname(__FILE__))),
            [],
            '1.0.1'
        );
        
        // Add dashicons for status icons
        wp_enqueue_style('dashicons');
    }

    public function addSettingsPage() {
        add_submenu_page(
            't4s-rewriter',
            'Settings - T4S Rewriter',
            'Settings',
            'manage_options',
            't4s-rewriter-settings',
            [$this, 'renderPage']
        );
    }

    public function registerSettings() {
        register_setting(
            't4s_rewriter_settings',
            't4s_rewriter_settings',
            [
                'sanitize_callback' => [$this, 'sanitizeSettings']
            ]
        );
        
        // Main Settings Section
        add_settings_section(
            't4s_rewriter_main_section',
            'Main Settings',
            null,
            't4s-rewriter-settings'
        );

        add_settings_field(
            't4s_rewriter_api_key',
            'API Key',
            [$this, 'renderApiKeyField'],
            't4s-rewriter-settings',
            't4s_rewriter_main_section'
        );

        add_settings_field(
            't4s_rewriter_language',
            'Default Language',
            [$this, 'renderLanguageField'],
            't4s-rewriter-settings',
            't4s_rewriter_main_section'
        );

        add_settings_field(
            't4s_rewriter_download_images',
            'Download Images',
            [$this, 'renderDownloadImagesField'],
            't4s-rewriter-settings',
            't4s_rewriter_main_section'
        );

        // API Information Section
        add_settings_section(
            't4s_rewriter_api_info_section',
            'API Information',
            [$this, 'renderApiInfoSection'],
            't4s-rewriter-settings'
        );
    }

    public function sanitizeSettings($input) {
        $sanitized = [];
        $current_settings = get_option('t4s_rewriter_settings', []);

        // Preserve writing_styles if not in current update
        if (!isset($input['writing_styles']) && isset($current_settings['writing_styles'])) {
            $sanitized['writing_styles'] = $current_settings['writing_styles'];
        } else {
            $sanitized['writing_styles'] = isset($input['writing_styles']) ? (array)$input['writing_styles'] : [];
        }

        // Sanitize other fields
        $sanitized['api_key'] = !empty($input['api_key']) ? sanitize_text_field($input['api_key']) : '';
        $sanitized['language'] = !empty($input['language']) ? sanitize_text_field($input['language']) : 'en';
        $sanitized['api_base_url'] = !empty($input['api_base_url']) ? esc_url_raw($input['api_base_url']) : 'https://api-rewriter.traffic4seo.com';
        $sanitized['download_images'] = isset($input['download_images']) ? (bool)$input['download_images'] : false;

        // Preserve any other existing settings not included in the update
        foreach ($current_settings as $key => $value) {
            if (!isset($sanitized[$key])) {
                $sanitized[$key] = $value;
            }
        }

        return $sanitized;
    }

    public function renderPage() {
        if (!current_user_can('manage_options')) {
            return;
        }

        if (isset($_GET['settings-updated'])) {
            add_settings_error(
                't4s_rewriter_messages',
                't4s_rewriter_message',
                'Settings Saved',
                'updated'
            );
        }

        // Pass necessary variables to template
        $settingsManager = $this->settingsManager;
        $apiClient = $this->apiClient;
        
        // Include template file
        require_once dirname(__FILE__) . '/templates/settings-page.php';
    }

    public function renderApiKeyField() {
        $settings = get_option('t4s_rewriter_settings');
        $api_key = isset($settings['api_key']) ? $settings['api_key'] : '';
        ?>
        <input type="text" 
               id="t4s_rewriter_api_key" 
               name="t4s_rewriter_settings[api_key]" 
               value="<?php echo esc_attr($api_key); ?>"
               class="regular-text"
        >
        <p class="description">Enter your API key for the rewriting service.</p>
        <?php
    }

    public function renderLanguageField() {
        $settings = get_option('t4s_rewriter_settings');
        $current_language = isset($settings['language']) ? $settings['language'] : 'en';
        $languages = $this->settingsManager->getAvailableLanguages();
        ?>
        <select id="t4s_rewriter_language"
                name="t4s_rewriter_settings[language]"
        >
            <?php foreach ($languages as $code => $name): ?>
                <option value="<?php echo esc_attr($code); ?>"
                        <?php selected($current_language, $code); ?>
                >
                    <?php echo esc_html($name); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <p class="description">Select the default language for rewritten content.</p>
        <?php
    }

    public function renderDownloadImagesField() {
        $settings = get_option('t4s_rewriter_settings');
        $download_images = isset($settings['download_images']) ? $settings['download_images'] : false;
        ?>
        <input type="checkbox"
               id="t4s_rewriter_download_images"
               name="t4s_rewriter_settings[download_images]"
               value="1"
               <?php checked($download_images, true); ?>
        >
        <p class="description">Download and save images locally when rewriting content.</p>
        <?php
    }

    public function renderApiBaseUrlField() {
        $settings = get_option('t4s_rewriter_settings');
        $api_base_url = isset($settings['api_base_url']) ? $settings['api_base_url'] : '';
        ?>
        <input type="text"
               id="t4s_rewriter_api_base_url"
               name="t4s_rewriter_settings[api_base_url]"
               value="<?php echo esc_attr($api_base_url); ?>"
               class="regular-text"
        >
        <p class="description">Enter the base URL for the API, e.g., https://api.example.com.</p>
        <?php
    }

    public function renderApiInfoSection() {
        $api_key = $this->settingsManager->getApiKey();
        if (!$api_key) {
            echo '<p>Please enter your API key to view account information.</p>';
            return;
        }

        try {
            $account_info = $this->apiClient->getAccountInfo();
            $usage_info = $this->apiClient->getUsage();
            ?>
            <table class="form-table api-info-table">
                <tr>
                    <th scope="row">Account Status</th>
                    <td>
                        <span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
                        Active
                    </td>
                </tr>
                <tr>
                    <th scope="row">Account Email</th>
                    <td><?php echo esc_html($account_info['email'] ?? 'N/A'); ?></td>
                </tr>
                <tr>
                    <th scope="row">Usage This Day</th>
                    <td>
                        <strong><?php echo esc_html($usage_info['used'] ?? 0); ?></strong> articles rewritten
                        <?php if (isset($usage_info['limit'])): ?>
                            out of <strong><?php echo esc_html($usage_info['limit']); ?></strong>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Last Checked</th>
                    <td><?php echo esc_html(current_time('mysql')); ?></td>
                </tr>
            </table>

            <?php if (isset($usage_info['used'], $usage_info['limit'])): ?>
                <div class="api-usage-bar">
                    <div class="api-usage-progress" style="width: <?php 
                        echo esc_attr(min(100, ($usage_info['used'] / $usage_info['limit'] * 100))); 
                    ?>%;"></div>
                </div>
            <?php endif;

        } catch (\Exception $e) {
            echo '<div class="notice notice-error inline"><p>Error fetching API information: ' . 
                 esc_html($e->getMessage()) . '</p></div>';
        }
    }
}