<?php

namespace T4S\Rewriter\Admin\Action;

use T4S\Rewriter\Job\Job;
use T4S\Rewriter\Job\JobManager;
use T4S\Rewriter\Cron\CronHandler;

class JobsActionHandler {
    private $jobManager;
    private $cronHandler;

    public function __construct(JobManager $jobManager, CronHandler $cronHandler) {
        $this->jobManager = $jobManager;
        $this->cronHandler = $cronHandler;
    }

    public function init() {
        add_action('admin_post_t4s_create_job', [$this, 'handleCreateJob']);
        add_action('admin_post_t4s_update_job', [$this, 'handleUpdateJob']);
        add_action('admin_post_t4s_delete_job', [$this, 'handleDeleteJob']);
        add_action('admin_post_t4s_fetch_rss_now', [$this, 'handleFetchRssNow']);
        add_action('admin_post_t4s_process_article_now', [$this, 'handleProcessArticleNow']);
        add_action('admin_post_t4s_process_specific_article', [$this, 'handleProcessSpecificArticle']);
    }

    public function handleCreateJob() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        check_admin_referer('t4s_create_job');

        // Validate input
        if (empty($_POST['rss_url'])) {
            wp_die('RSS URL is required.');
        }

        if (empty($_POST['category_ids']) || !is_array($_POST['category_ids'])) {
            wp_die('At least one category is required.');
        }

        if (empty($_POST['writing_style'])) {
            wp_die('Writing style is required.');
        }

        // Sanitize input
        $rss_url = esc_url_raw($_POST['rss_url']);
        if (!$rss_url) {
            wp_die('Invalid RSS URL format.');
        }

        // Sanitize category IDs
        $category_ids = array_map('intval', $_POST['category_ids']);
        $category_ids = array_filter($category_ids); // Remove any 0 values
        if (empty($category_ids)) {
            wp_die('Invalid category selection.');
        }

        try {
            $job = new Job([
                'rss_url' => $rss_url,
                'category_ids' => json_encode($category_ids),
                'writing_style' => sanitize_text_field($_POST['writing_style']),
                'post_status' => in_array($_POST['post_status'], ['draft', 'publish']) ? $_POST['post_status'] : 'draft',
                'is_active' => isset($_POST['is_active']) ? 1 : 0,
                'daily_post_count' => isset($_POST['daily_post_count']) ? max(0, (int)$_POST['daily_post_count']) : 0,
                'hourly_post_count' => isset($_POST['hourly_post_count']) ? max(0, (int)$_POST['hourly_post_count']) : 0,
                'is_ascending' => isset($_POST['is_ascending']) ? 1 : 0,
                'require_featured_image' => isset($_POST['require_featured_image']) ? 1 : 0
            ]);

            // Attempt to create job
            $job_id = $this->jobManager->createJob($job);
            
            if ($job_id) {
                $this->addNotice('Job created successfully.', 'updated');
                wp_redirect(admin_url('admin.php?page=t4s-rewriter-jobs'));
                exit;
            }
        } catch (\Exception $e) {
            error_log('T4S Rewriter job creation error: ' . $e->getMessage());
            wp_die(
                'Error creating job: ' . esc_html($e->getMessage()),
                'Job Creation Error',
                ['back_link' => true]
            );
        }
    }

    public function handleUpdateJob() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        check_admin_referer('t4s_update_job');

        if (empty($_POST['job_id'])) {
            wp_die('Job ID is required.');
        }

        try {
            // Get existing job to preserve some values
            $existing_job = $this->jobManager->getJob((int)$_POST['job_id']);
            if (!$existing_job) {
                throw new \Exception('Job not found.');
            }

            // Validate and sanitize category IDs
            if (empty($_POST['category_ids']) || !is_array($_POST['category_ids'])) {
                wp_die('At least one category is required.');
            }

            $category_ids = array_map('intval', $_POST['category_ids']);
            $category_ids = array_filter($category_ids); // Remove any 0 values
            if (empty($category_ids)) {
                wp_die('Invalid category selection.');
            }

            $job = new Job([
                'id' => (int)$_POST['job_id'],
                'rss_url' => esc_url_raw($_POST['rss_url']),
                'category_ids' => json_encode($category_ids),
                'writing_style' => sanitize_text_field($_POST['writing_style']),
                'post_status' => in_array($_POST['post_status'], ['draft', 'publish']) ? $_POST['post_status'] : 'draft',
                'is_active' => isset($_POST['is_active']) ? 1 : 0,
                'articles_processed' => $existing_job->getArticlesProcessed(),
                'last_run' => $existing_job->getLastRun(),
                'daily_post_count' => isset($_POST['daily_post_count']) ? max(0, (int)$_POST['daily_post_count']) : 0,
                'hourly_post_count' => isset($_POST['hourly_post_count']) ? max(0, (int)$_POST['hourly_post_count']) : 0,
                'is_ascending' => isset($_POST['is_ascending']) ? 1 : 0,
                'require_featured_image' => isset($_POST['require_featured_image']) ? 1 : 0
            ]);

            if ($this->jobManager->updateJob($job)) {
                $this->addNotice('Job updated successfully.', 'updated');
                wp_redirect(admin_url('admin.php?page=t4s-rewriter-jobs'));
                exit;
            }
        } catch (\Exception $e) {
            error_log('T4S Rewriter job update error: ' . $e->getMessage());
            wp_die(
                'Error updating job: ' . esc_html($e->getMessage()),
                'Job Update Error',
                ['back_link' => true]
            );
        }
    }

    public function handleDeleteJob() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        check_admin_referer('t4s_delete_job');

        $job_id = (int)$_POST['job_id'];
        if (!$job_id) {
            wp_die('Invalid job ID.');
        }

        try {
            if ($this->jobManager->deleteJob($job_id)) {
                $this->addNotice('Job deleted successfully.', 'updated');
                wp_redirect(admin_url('admin.php?page=t4s-rewriter-jobs'));
                exit;
            }
        } catch (\Exception $e) {
            error_log('T4S Rewriter job deletion error: ' . $e->getMessage());
            wp_die(
                'Error deleting job: ' . esc_html($e->getMessage()),
                'Job Deletion Error',
                ['back_link' => true]
            );
        }
    }

    public function handleFetchRssNow() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        check_admin_referer('t4s_fetch_rss_now');

        $job_id = (int)$_POST['job_id'];
        $job = $this->jobManager->getJob($job_id);

        if (!$job) {
            wp_die('Job not found');
        }

        try {
            // Process only this job's RSS feed
            $this->cronHandler->processJobs([$job]);
            $this->addNotice('RSS feed fetched successfully.', 'updated');
        } catch (\Exception $e) {
            $this->addNotice('Error fetching RSS: ' . $e->getMessage(), 'error');
        }

        wp_redirect(admin_url('admin.php?page=t4s-rewriter-jobs&action=view&job_id=' . $job_id));
        exit;
    }

    public function handleProcessArticleNow() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        check_admin_referer('t4s_process_article_now');

        try {
            // Process one pending article
            $processed = $this->cronHandler->processOneArticle();
            
            if ($processed) {
                $this->addNotice('Article processed successfully.', 'updated');
            } else {
                $this->addNotice('No pending articles to process or API limit reached.', 'error');
            }
        } catch (\Exception $e) {
            $this->addNotice('Error processing article: ' . $e->getMessage(), 'error');
        }

        $job_id = (int)$_POST['job_id'];
        wp_redirect(admin_url('admin.php?page=t4s-rewriter-jobs&action=view&job_id=' . $job_id));
        exit;
    }

    public function handleProcessSpecificArticle() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }

        check_admin_referer('t4s_process_specific_article');

        $article_id = (int)$_POST['article_id'];

        try {
            $processed = $this->cronHandler->processOneArticle($article_id);
            
            if ($processed) {
                $this->addNotice('Article processed successfully.', 'updated');
            } else {
                $this->addNotice('Could not process article at this time.', 'error');
            }
        } catch (\Exception $e) {
            $this->addNotice('Error processing article: ' . $e->getMessage(), 'error');
        }

        $job_id = (int)$_POST['job_id'];
        wp_redirect(admin_url('admin.php?page=t4s-rewriter-jobs&action=view&job_id=' . $job_id));
        exit;
    }

    private function addNotice($message, $type = 'updated') {
        set_transient('t4s_rewriter_admin_notices', [
            [
                'type' => $type,
                'message' => $message
            ]
        ], 30);
    }
}